<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WPB_Blacklist_Basic {

    /**
     * Option key used to store the phone list.
     *
     * @var string
     */
    private $option_key = 'wpb_blacklist_basic_numbers';

    public function __construct() {
        // Admin.
        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
        add_action( 'admin_init', array( $this, 'handle_form_submit' ) );

        // Checkout.
        add_filter( 'woocommerce_checkout_fields', array( $this, 'require_phone' ) );
        add_action( 'woocommerce_checkout_process', array( $this, 'validate_checkout_phone' ) );
    }

    /**
     * Get blacklist array from options.
     *
     * @return array
     */
    private function get_blacklist() {
        $list = get_option( $this->option_key, array() );
        if ( ! is_array( $list ) ) {
            $list = array();
        }
        return $list;
    }

    /**
     * Save blacklist in options, after normalization and deduplication.
     *
     * @param array $phones Raw phone numbers.
     */
    private function save_blacklist( array $phones ) {
        $clean = array();

        foreach ( $phones as $phone ) {
            $phone = $this->normalize_phone( $phone );
            if ( $phone ) {
                $clean[ $phone ] = $phone;
            }
        }

        update_option( $this->option_key, array_values( $clean ), false );
    }

    /**
     * Basic GR phone normalization (no external libs).
     *
     * @param string $phone Raw phone.
     * @return string Normalized phone or empty string.
     */
    private function normalize_phone( $phone ) {
        $phone = preg_replace( '/\D+/', '', (string) $phone );

        if ( '' === $phone ) {
            return '';
        }

        // 10-digit Greek numbers starting with 2 or 6 → prepend 30.
        if ( strlen( $phone ) === 10 && ( '2' === $phone[0] || '6' === $phone[0] ) ) {
            $phone = '30' . $phone;
        }

        // If still 10 digits, prepend 30 as a fallback.
        if ( strlen( $phone ) === 10 ) {
            $phone = '30' . $phone;
        }

        // Add leading + for E.164-like format.
        if ( 0 === strpos( $phone, '30' ) ) {
            $phone = '+' . $phone;
        }

        return $phone;
    }

    /**
     * Check if phone is blacklisted.
     *
     * @param string $phone Raw phone.
     * @return bool
     */
    private function is_blacklisted( $phone ) {
        $normalized = $this->normalize_phone( $phone );
        if ( '' === $normalized ) {
            return false;
        }

        $list = $this->get_blacklist();

        return in_array( $normalized, $list, true );
    }

    /* ==================== Admin UI ==================== */

    /**
     * Add submenu page under WooCommerce.
     */
    public function add_admin_menu() {
        add_submenu_page(
            'woocommerce',
            __( 'Phone Blacklist (Basic)', 'phone-blacklist-basic' ),
            __( 'Phone Blacklist', 'phone-blacklist-basic' ),
            'manage_woocommerce',
            'wpb-phone-blacklist-basic',
            array( $this, 'render_admin_page' )
        );
    }

    /**
     * Handle form submission for blacklist.
     */
    public function handle_form_submit() {
        if ( ! isset( $_POST['wpb_blacklist_basic_nonce'] ) ) {
            return;
        }

        $nonce = sanitize_text_field( wp_unslash( $_POST['wpb_blacklist_basic_nonce'] ) );

        if ( ! wp_verify_nonce( $nonce, 'wpb_blacklist_basic_save' ) ) {
            return;
        }

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            return;
        }

        $raw = isset( $_POST['wpb_blacklist_numbers'] )
            ? sanitize_textarea_field( wp_unslash( $_POST['wpb_blacklist_numbers'] ) )
            : '';

        $lines  = preg_split( '/\r\n|\r|\n/', (string) $raw );
        $phones = array();

        foreach ( $lines as $line ) {
            $line = trim( $line );
            if ( '' !== $line ) {
                $phones[] = $line;
            }
        }

        $this->save_blacklist( $phones );

        add_settings_error(
            'wpb_blacklist_basic_notice',
            'wpb_blacklist_basic_notice',
            __( 'Blacklist updated.', 'phone-blacklist-basic' ),
            'updated'
        );
    }

    /**
     * Render admin page.
     */
    public function render_admin_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'phone-blacklist-basic' ) );
        }

        $list  = $this->get_blacklist();
        $value = implode( "\n", $list );

        settings_errors( 'wpb_blacklist_basic_notice' );
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Phone Blacklist for WooCommerce (Basic)', 'phone-blacklist-basic' ); ?></h1>
            <p><?php esc_html_e( 'Add one phone number per line. Any order using one of these phone numbers will be blocked at checkout.', 'phone-blacklist-basic' ); ?></p>

            <form method="post">
                <?php wp_nonce_field( 'wpb_blacklist_basic_save', 'wpb_blacklist_basic_nonce' ); ?>

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">
                            <label for="wpb_blacklist_numbers">
                                <?php esc_html_e( 'Blacklisted phone numbers', 'phone-blacklist-basic' ); ?>
                            </label>
                        </th>
                        <td>
                            <textarea id="wpb_blacklist_numbers"
                                      name="wpb_blacklist_numbers"
                                      rows="12"
                                      cols="50"
                                      class="large-text code"
                                      placeholder="+3069XXXXXXXX"><?php echo esc_textarea( $value ); ?></textarea>
                            <p class="description">
                                <?php esc_html_e( 'One phone number per line. Formats like +3069XXXXXXXX, 69XXXXXXXX, 2XXXXXXXXX will be normalized automatically.', 'phone-blacklist-basic' ); ?>
                            </p>
                        </td>
                    </tr>
                </table>

                <?php submit_button( __( 'Save blacklist', 'phone-blacklist-basic' ) ); ?>
            </form>
        </div>
        <?php
    }

    /* ==================== Checkout ==================== */

    /**
     * Make billing phone required.
     *
     * @param array $fields Checkout fields.
     * @return array
     */
    public function require_phone( $fields ) {
        if ( isset( $fields['billing']['billing_phone'] ) ) {
            $fields['billing']['billing_phone']['required'] = true;
        }
        return $fields;
    }

    /**
     * Validate phone at checkout.
     */
    public function validate_checkout_phone() {
        // WooCommerce checkout already verifies a nonce, but we add explicit check for Plugin Check compliance.
        if ( ! isset( $_POST['woocommerce-process-checkout-nonce'] ) ) {
            return;
        }

        $nonce = sanitize_text_field( wp_unslash( $_POST['woocommerce-process-checkout-nonce'] ) );

        if ( ! wp_verify_nonce( $nonce, 'woocommerce-process_checkout' ) ) {
            return;
        }

        if ( ! isset( $_POST['billing_phone'] ) ) {
            return;
        }

        $phone = sanitize_text_field( wp_unslash( $_POST['billing_phone'] ) );

        if ( '' === $phone ) {
            wc_add_notice(
                __( 'A phone number is required to complete your order.', 'phone-blacklist-basic' ),
                'error'
            );
            return;
        }

        if ( $this->is_blacklisted( $phone ) ) {
            wc_add_notice(
                __( 'Your order cannot be completed. Please contact customer support.', 'phone-blacklist-basic' ),
                'error'
            );
        }
    }
}
